<?php
require_once APPPATH . 'third_party/vendor/autoload.php';

defined('BASEPATH') or exit('No direct script access allowed');

class Bot_model extends MY_Model
{

    public function __construct()
    {
        parent::__construct();
        $this->load->model('azureblob_model');
        $this->load->model('bhashini_model');
    }

    /**
     * Get screening bot settings for a given screeningId.
     * Used in Bot controller: index(), ajaxPrepare()
     */
    public function getSettings($screeningId)
    {
        $this->db->select('screeningId, interviewType, isContextual, gpt, proctoring, voiceType, 
                          vernacularResponse, interviewLanguage, interviewDuration, isFraudPage, isAttirePage,isVoiceMatchPage');
        $this->db->from('screening_bot_settings as sbs');
        $this->db->where('sbs.screeningId', $screeningId);
        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Get the list of questions to generate for the interview based on settings.
     * Used in Bot controller: ajaxPrepare()
     */
    public function getQuestionsToGenerate($getSettings)
    {
        $this->db->select('id, question');
        $this->db->from('questions_custom as qc');
        $this->db->where('qc.screeningId', $getSettings['screeningId']);
        $this->db->order_by('qc.id', 'ASC');
        $query = $this->db->get();
        $questions = $query->result_array();
        return $questions;
    }

    /**
     * Get already generated questions for the interview (with audio URLs).
     * Used in Bot controller: ajaxPrepare()
     */
    public function getQuestions($getSettings)
    {
        //pre($getSettings);die;
        $langId = $this->getLanguageId($getSettings);
        $this->db->select('id, question, audio');
        $this->db->from('questions_vernacular as qv');
        $this->db->where('qv.screeningId', $getSettings['screeningId']);
        $this->db->where('qv.langId', $langId);
        $this->db->order_by('qv.id', 'ASC');
        $query = $this->db->get();
        $questions = $query->result_array();
        // echo $this->db->last_query();die;

        foreach ($questions as &$row) {
            $row['audioUrl'] = $this->azureblob_model->generateSASToken(QUESTIONAUDIOCONTAINER, $row['audio']);
        }
        unset($row);
        return $questions;
    }

    /**
     * Get the language ID for the interview, based on settings and candidate.
     * Used in: getQuestions()
     */
    public function getLanguageId($getSettings)
    {
        if ($getSettings['vernacularResponse'] == 'off') {
            $langId = $getSettings['interviewLanguage'];
        } else {
            $this->db->select('candidateLanguageId');
            $this->db->from('bot_interview');
            $this->db->where('ref', "SID" . $this->session->userdata('commonId'));
            $query = $this->db->get()->row_array();
            $langId = $query['candidateLanguageId'];
        }
        return $langId;
    }

    /**
     * Insert a new bot interview record (consent, language, etc).
     * Not directly used in Bot controller (likely used elsewhere in the project).
     */
    public function insertIntoBotSettings($data)
    {
        $commonId = $this->session->userdata('commonId');
        $params = array(
            'screeningId' => $data['screeningId'],
            'candidateId' => $data['candidateId'],
            'resumeId' => $data['resumeId'],
            'ref' => 'SID' . $commonId,
            'created' => date('Y-m-d H:i:s'),
            'isConsent' => 1,
            'language' => isset($data['language']) ? $data['language'] : 0
        );
        return $this->db->insert('bot_interview', $params);
    }

    /**
     * Check if the candidate has already given the interview.
     * Not directly used in Bot controller (likely used elsewhere in the project).
     */
    public function isGivenInterview($data)
    {
        $this->db->select('COUNT(*) as cnt');
        $this->db->from('bot_interview');
        $this->db->where(array(
            'candidateId' => $data['candidateId'],
            'resumeId' => $data['resumeId'],
            'screeningId' => $data['screeningId'],
            'isVal' => 1
        ));
        $query = $this->db->get();
        $result = $query->row_array();
        return isset($result['cnt']) ? $result['cnt'] : 0;
    }

    /**
     * Update the interview status (e.g., END, IN_PROGRESS).
     * Not directly used in Bot controller (likely used elsewhere in the project).
     */
    public function updateInterviewStatus($status, $commonId)
    {
        $params = array(
            'status' => $status,
            'modified' => date('Y-m-d H:i:s')
        );

        if ($status === 'END') {
            $params['isCron'] = 0;
        }

        $this->db->where('ref', 'SID' . $commonId);
        return $this->db->update('bot_interview', $params);
    }


    /**
     * Get all available languages from the fill_gpt_languages table.
     * Used in Bot controller: index()
     */
    public function getLanguages($id = "")
    {
        $this->db->select('*');
        $this->db->from('fill_gpt_languages');
        $this->db->where('isActive', 1);
        $this->db->order_by("language", "asc");

        if (!empty($id)) {
            $this->db->where('id', $id);
            $query = $this->db->get();
            return $query->row_array();
        } else {
            $query = $this->db->get();
            return $query->result_array();
        }
    }

    /**
     * Generate and insert the next question for the interview.
     * Used in Bot controller: ajaxPrepare()
     * @param array $getSettings - Interview settings
     * @param array $question - Question data to insert
     * Fetches prompt from prompts table, then inserts question into questions_vernacular.
     */
    // public function generateNextQuestion($getSettings, $question)
    // {
    //     // Fetch prompt from fill_prompts table based on question or screeningId
    //     $langId = $this->getLanguageId($getSettings);
    //     $language = $this->getLanguages($langId);
    //     //pre($language);die;
    //     $this->db->select('prompt');
    //     $this->db->from('fill_prompts');
    //     $this->db->where('title', 'Translate_and_Humanize_Questions');
    //     $promptRow = $this->db->get()->row_array();
    //     //pre($promptRow); die;// Debugging line, remove in production
    //     $prompt = isset($promptRow['prompt']) ? $promptRow['prompt'] : '';
    //     $replace = array(
    //         '{{desired_language}}' => $language['language'],
    //         '{{gender}}' => 'Female'
    //     );
    //     $prompt = str_replace(array_keys($replace), array_values($replace), $prompt);        // Prepare data for insertion
    //     $generatedQuestion = $this->callEngineLLM($prompt);

    //     $data = array(
    //         'screeningId' => $getSettings['screeningId'],
    //         'question' => $question['question'],
    //         'audio' => isset($question['audio']) ? $question['audio'] : null,
    //         'prompt' => $prompt,
    //         'created' => date('Y-m-d H:i:s')
    //     );

    //     return $this->db->insert('questions_vernacular', $data);
    // }


    public function countQuestionsWithAudio($getSettings)
    {
        $langId = $this->getLanguageId($getSettings);
        $this->db->from('questions_vernacular');
        $this->db->where('screeningId', $getSettings['screeningId']);
        $this->db->where('langId', $langId);
        $this->db->where("audio <> ''", null, false);
        // echo $this->db->last_query();die;
        return $this->db->count_all_results();
    }

    // Generate audio for the next question without audio
    public function generateAudioForQuestion($getSettings, $index = '')
    {
        $langId = $this->getLanguageId($getSettings);
        $this->db->from('questions_vernacular');
        $this->db->where('screeningId', $getSettings['screeningId']);
        $this->db->where('langId', $langId);
        $this->db->where("(audio IS NULL OR audio = '')", null, false);
        $this->db->order_by('id', 'ASC');
        $this->db->limit(1);
        $arrQuestion = $this->db->get()->row_array();
        // echo $this->db->last_query();die;
        // Debugging line, remove in production
        if ($arrQuestion) {
            // Call your TTS/audio generation logic here, e.g.:
            $audioFile = $this->callEngineTTS($arrQuestion, $getSettings);
            // Update the question with the audio file path
            $this->db->where('id', $arrQuestion['id']);
            $this->db->update('questions_vernacular', ['audio' => $audioFile]);
        }
    }

    /**
     * This function is used to generate questions from english to others languages.
     */
    public function generateAllQuestions($getSettings, $questionsArray)
    {
        $langId = $this->getLanguageId($getSettings);
        $language = $this->getLanguages($langId);
        if ($language['type'] == 'BHASHINI') { 
            $this->bhashini_model->generateAllQuestionsBhashini($getSettings, $questionsArray, $language);
        } else {
            $this->generateAllQuestionsOthers($getSettings, $questionsArray, $language);
        }

    }

    public function generateAllQuestionsOthers($getSettings, $questionsArray, $language)
    {
        //pre($getSettings);die;

        //$language['language'] = 'Hindi';
        // Prepare prompt for all questions
        $questionsText = array_column($questionsArray, 'question');
        $prompt = $this->getPromptsByTitle('Translate_and_Humanize_Questions');
        $replace = array(
            '{{desired_language}}' => $language['language'],
            '{{gender}}' => 'Female',
            '{{questions}}' => implode("\n", $questionsText)
        );
        $prompt = str_replace(array_keys($replace), array_values($replace), $prompt);        // Prepare data for insertion
        try {
            $data = $this->callEngineLLM($prompt);
            if (isset($data['response'])) {
                $generatedQuestions = $data['response'];
                //echo $translatedText; die;// ✅ This will now show Hindi characters
            } else {
                echo "Invalid response format.";
            }
            // Step 1: Explode by ###
            $generatedQuestions = explode("###", $generatedQuestions);
            // Step 2: Clean up extra space and empty entries
            $generatedQuestions = array_filter(array_map('trim', $generatedQuestions));
            // Step 3: Map with qId
            $mapped = [];
            foreach (array_values($generatedQuestions) as $index => $generated) {
                if (isset($questionsArray[$index]['id'])) {
                    $data = [
                        'originalQuestionId' => $questionsArray[$index]['id'],
                        'screeningId' => $getSettings['screeningId'],
                        'langId' => $language['id'],
                        'question' => $generated, // or whatever key your API returns
                        'created' => date('Y-m-d H:i:s')
                    ];
                    $this->db->insert('questions_vernacular', $data);
                }
            }

        } catch (\Exception $e) {
            echo "An exception occurred: " . $e->getMessage();
            return null;
        }
    }

    /**
     * Summary of getPromptsByTitle
     * @param mixed $title
     */
    public function callEngineLLM($prompt)
    {
        try {
            $client = new \GuzzleHttp\Client();

            $engineUrl = ENGINEURL . "/api/llm";
            $llmData = [
                'engine' => 'openai',
                'prompt' => $prompt, // this can be a string or heredoc or from file
            ];
            //echo $prompt;die;
            $response = $client->request('POST', $engineUrl, [
                'json' => $llmData
            ]);

            // Get body as string
            $body = $response->getBody()->getContents();

            // Decode JSON
            $data = json_decode($body, true); // true => associative array
            return $data;
            // Now access the 'response' field
           
        } catch (\Exception $e) {
            echo "An exception occurred: " . $e->getMessage();
            return null;
        }
    }

    public function callEngineTTS($arrQuestion, $getSettings)
    {
        $langId = $this->getLanguageId($getSettings);
        $language = $this->getLanguages($langId);

        try {
            if($language['type'] == 'BHASHINI') {
                $data = $this->bhashini_model->callBhashiniTTS($arrQuestion, $getSettings, $language);
                if(!empty($data)){
                    $fileName = $data['fileName'];
                    $path = $data['path'];
                }
            } else{
                $client = new \GuzzleHttp\Client();

                $engineUrl = ENGINEURL . "/api/tts";
                $llmData = [
                    'text' => $arrQuestion['question'],
                    'engine' => 'elevenlabs',
                    //'voice' => $getSettings['voiceType'], // Rachel, etc.
                    'voice' => 'EaBs7G1VibMrNAuz2Na7', // Here we need to send voice id //Monika-EaBs7G1VibMrNAuz2Na7
                    'name' => $arrQuestion['screeningId'] . '_' . $arrQuestion['langId'] . '_' . time(), // this can be a string or heredoc or from file
                ];
                // pre($llmData);die;
                //echo $prompt;die;
                $response = $client->request('POST', $engineUrl, [
                    'json' => $llmData
                ]);

                // Get body as string
                $body = $response->getBody()->getContents();

                // Decode JSON
                $data = json_decode($body, true); // true => associative array
                $fileName = $data['filename'];
                $path = str_replace('bot', 'engine', FCPATH);
                $path = $path . '/temp/questions_audios/'. $fileName;
             }
            //pre($data);die;
            // Now access the 'response' field
            if (isset($fileName)) {
                // Upload to Azure Blob Storage
                if (file_exists($path)) {
                    $content = fopen($path, "r"); //For copy file use localpath and for upload file use tmppath.
                    $response = $this->azureblob_model->azureBlobUpload(QUESTIONAUDIOCONTAINER, $fileName, $content);
                    //echo $response;
                    if ($response == 'SUCCESS') {
                        unlink($path);
                    }
                }
                // Return the file name for further processing  
                return $fileName;
            } else {
                echo "Invalid response format.";
            }
        } catch (\Exception $e) {
            echo "An exception occurred: " . $e->getMessage();
            return null;
        }
    }

    /**
     * Function is used for calling the Speech-to-Text engine.
     */
    public function callEngineSTT($filePath, $iso){
        $client = new \GuzzleHttp\Client();
        $engineUrl = ENGINEURL . "/api/stt";

        // Prepare the request to the STT API
        //start timer for performance measurement
        
        $response = $client->request('POST', $engineUrl, [
            'json' => [
                'file_path' => $filePath,
                'engine' => 'whisper',
                'iso' => $iso // Language ISO code
            ]
        ]);
        echo "Status Code: " . $response->getStatusCode() . "\n";
        echo "Response:\n" . $response->getBody();
        $arrData = json_decode($response->getBody(), true);
        return $arrData;
    }

    /**
     * Check if the candidate has already given the interview.
     * Used in Bot controller: systemCheck()
     */
    public function getFromBotInterview($data)
    {
        $this->db->select('*');
        $this->db->from('bot_interview');
        $this->db->where('candidateId', $data['candidateId']);
        $this->db->where('resumeId', $data['resumeId']);
        $this->db->where('screeningId', $data['screeningId']);
        $this->db->where('isVal', 1);
        $query = $this->db->get()->row_array();
        return $query ? $query : [];
    }
}
