<?php $this->load->view('templates/header'); ?>
<div class="voice-container">
    <h2>Identity Verification</h2>
    <p class="voice-instruction">Please record your voice sample by speaking below statement clearly &amp; loudly</p>
    <p class="voice-statement">
        This morning, the hills were covered in a soft mist, and the air felt cooler than usual for this time of year. A gentle breeze moved through the trees, carrying the scent of rain from the night before. It reminded me how small changes in climate can shift the rhythm of nature, from when flowers bloom to how rivers flow.
    </p>
    <center><canvas id="eq" width="500" height="80"></canvas></center>
    <button id="recordBtn" class="voice-record">Click to Record</button>
    <button id="stopBtn" class="voice-stop" style="display:none;">Stop Recording</button>
    <audio id="audioPlayback"></audio>
    <div class="audio-mini" id="miniPlayer" style="display:none;">
        <button class="audio-btn" id="audioToggle" aria-label="Play" aria-pressed="false">
            ▶
        </button>
        <span class="audio-time" id="audioTime">0:00 / 0:00</span>
    </div>
    <div class="btn-group">
        <button id="backBtn" class="voice-back">Back</button>
        <button id="submitBtn" class="get-started-btn" disabled>Submit</button>
    </div>
</div>
<script>

let mediaRecorder;
let audioChunks = [];
const recordBtn = document.getElementById('recordBtn');
const stopBtn = document.getElementById('stopBtn');
const submitBtn = document.getElementById('submitBtn');
const audioPlayback = document.getElementById('audioPlayback');
const BASE_URL = "<?php echo base_url(); ?>";
recordBtn.onclick = async function () {
    audioChunks = [];
    let stream = await navigator.mediaDevices.getUserMedia({ audio:true });
    mediaRecorder = new MediaRecorder(stream);

    mediaRecorder.ondataavailable = (e) => {
        audioChunks.push(e.data);
    };

    mediaRecorder.onstop = (e) => {
        const audioBlob = new Blob(audioChunks, { type: 'audio/webm' });
        const audioUrl = URL.createObjectURL(audioBlob);
        audioPlayback.src = audioUrl;
        document.getElementById('miniPlayer').style.display = 'block';
        submitBtn.disabled = false;
        // Save the blob for later submit
        submitBtn.audioBlob = audioBlob;
    };

    mediaRecorder.start();
    recordBtn.style.display = 'none';
    stopBtn.style.display = 'inline-block';
};

stopBtn.onclick = function () {
    if(mediaRecorder && mediaRecorder.state !== "inactive") {
        mediaRecorder.stop();
    }
    recordBtn.style.display = 'inline-block';
    stopBtn.style.display = 'none';
    stopEqualizer();
};

(function initMiniAudio() {
  const audio = document.getElementById('audioPlayback');
  const btn = document.getElementById('audioToggle');
  const timeEl = document.getElementById('audioTime');

  // Utility: mm:ss
  function fmt(t) {
    if (!isFinite(t) || t < 0) return '0:00';
    const m = Math.floor(t / 60);
    const s = Math.floor(t % 60).toString().padStart(2, '0');
    return `${m}:${s}`;
  }

  function updateTime() {
    timeEl.textContent = `${fmt(audio.currentTime)} / ${fmt(audio.duration)}`;
  }

  function setPlayingUI(isPlaying) {
    btn.textContent = isPlaying ? '❚❚' : '▶';
    btn.setAttribute('aria-label', isPlaying ? 'Pause' : 'Play');
    btn.setAttribute('aria-pressed', String(isPlaying));
  }

  // Load metadata to show total duration
  if (audio.readyState >= 1) updateTime();
  audio.addEventListener('loadedmetadata', updateTime);
  audio.addEventListener('timeupdate', updateTime);
  audio.addEventListener('ended', () => setPlayingUI(false));
  audio.addEventListener('play', () => setPlayingUI(true));
  audio.addEventListener('pause', () => setPlayingUI(false));

  // Button toggles play/pause
  btn.addEventListener('click', async () => {
    try {
      if (audio.paused) {
        await audio.play();
      } else {
        audio.pause();
      }
    } catch (err) {
      console.error('Playback error:', err);
    }
  });

  // Optional: space/enter support when button focused (already works by default),
  // but allow keyboard K to toggle when container focused:
  document.getElementById('miniPlayer').addEventListener('keydown', (e) => {
    if (e.key.toLowerCase() === 'k') {
      btn.click();
    }
  });
})();


async function checkVoiceQuality() {
    try {
        const response = await fetch(`${BASE_URL}analyze/checkVoiceQuality`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
              status: 'pending'
            }) // Empty body since commonId comes from session
        });

        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }

        const result = await response.json();
        return result;
    } catch (error) {
        console.error('Voice quality check failed:', error);
        throw error;
    }
}

function displayQualityResults(qualityResult) {
    // Create a modal or alert to show results
    const resultMessage = `
Voice Quality Analysis:
Rating: ${qualityResult.rating}
Loudness: ${qualityResult.lufs} LUFS
Signal-to-Noise Ratio: ${qualityResult.snr_db} dB
Clarity Score: ${qualityResult.clarity_score}
Loudness Score: ${qualityResult.loudness_score}
Notes: ${qualityResult.notes}
    `;
    
    // You can replace this with a nicer modal
    alert(resultMessage);
    
    // Or log to console for debugging
    console.log('Voice Quality Results:', qualityResult);
}


submitBtn.onclick = async function () {
    let formData = new FormData();
    formData.append('audio', submitBtn.audioBlob, 'voice_sample.webm');

    submitBtn.disabled = true;
    submitBtn.textContent = "Uploading...";

    fetch(`${BASE_URL}analyze/saveVoiceSample`, {
        method: 'POST',
        body: formData
    }).then(res => res.json())
      .then(async response => {
        if(response.status == 'success'){
            submitBtn.textContent = "Analyzing Quality...";
            
            // Check voice quality after successful upload
            try {
                const qualityResult = await checkVoiceQuality();
                
                if (qualityResult.error) {
                    throw new Error(qualityResult.error);
                }
                
                // Show results based on quality rating
                const rating = qualityResult.rating;
                let backgroundColor = "#28a745"; // green
                
                if (rating === 'poor') {
                    backgroundColor = "#dc3545"; // red
                } else if (rating === 'moderate') {
                    backgroundColor = "#ffc107"; // yellow
                }
                
                submitBtn.textContent = "Submitted!";
                submitBtn.style.background = backgroundColor;
                
                // Show detailed results
                displayQualityResults(qualityResult);
                
            } catch (qualityError) {
                submitBtn.textContent = "Upload Complete";
                submitBtn.style.background = "#ffc107";
                alert("Quality check failed: " + qualityError.message);
            }

        } else {
            submitBtn.disabled = false;
            submitBtn.textContent = "Submit";
            alert("Upload failed: " + response.message);
        }
      })
      .catch(err => {
        submitBtn.disabled = false;
        submitBtn.textContent = "Submit";
        alert("Upload error: " + err.toString());
      });
};



// async function startEqualizer({
//   canvasId = 'eq',
//   barCount = 12,
//   barWidth = 10,
//   barGap = 8,
//   barColor = '#1da1f2',       // blue
//   bgColor = 'transparent',
//   cornerRadius = 4,
//   smoothing = 0.7,            // 0..1; higher = smoother
//   fftSize = 2048              // higher FFT for better banding
// } = {}) {
//   const canvas = document.getElementById(canvasId);
//   if (!canvas) throw new Error('Canvas not found');
//   const ctx = canvas.getContext('2d');

//   // Handle HiDPI scaling
//   const cssW = canvas.clientWidth || canvas.width;
//   const cssH = canvas.clientHeight || canvas.height;
//   const dpr = Math.max(1, window.devicePixelRatio || 1);
//   canvas.width = Math.round(cssW * dpr);
//   canvas.height = Math.round(cssH * dpr);
//   ctx.scale(dpr, dpr);

//   // Prepare audio input
//   const stream = await navigator.mediaDevices.getUserMedia({ audio: true, video: false });
//   const audioCtx = new (window.AudioContext || window.webkitAudioContext)();
//   const source = audioCtx.createMediaStreamSource(stream);
//   const analyser = audioCtx.createAnalyser();
//   analyser.fftSize = fftSize;
//   analyser.smoothingTimeConstant = 0.6; // analyzer's own smoothing
//   source.connect(analyser);

//   const freqData = new Uint8Array(analyser.frequencyBinCount);
//   const barHeights = new Array(barCount).fill(0);

//   // Build log-spaced bands across the speech range
//   function makeLogBands(nBars, nBins, sampleRate = 44100) {
//     const fNyquist = sampleRate / 2;
//     const hzPerBin = fNyquist / nBins;
//     const fMin = 80;
//     const fMax = 8000;
//     const bands = [];
//     for (let i = 0; i < nBars; i++) {
//       const t0 = i / nBars;
//       const t1 = (i + 1) / nBars;
//       const f0 = fMin * Math.pow(fMax / fMin, t0);
//       const f1 = fMin * Math.pow(fMax / fMin, t1);
//       const b0 = Math.max(0, Math.floor(f0 / hzPerBin));
//       const b1 = Math.min(nBins - 1, Math.ceil(f1 / hzPerBin));
//       bands.push([b0, b1]);
//     }
//     return bands;
//   }

//   // Perceptual mapping: 0..255 -> 0..1 with log-like response + soft knee
//   function magToLevel(v, knee = 0.2) {
//     const x = v / 255;                  // 0..1
//     const y = Math.log10(1 + 9 * x);    // 0..1, compresses dynamic range
//     return (y < knee) ? (y / knee) * 0.25 : 0.25 + (y - knee) * (0.75 / (1 - knee));
//   }

//   const noiseFloor = 0.02; // suppress tiny motions from background noise
//   let logBands = makeLogBands(barCount, freqData.length, audioCtx.sampleRate);

//   function roundRect(x, y, w, h, r) {
//     const rr = Math.min(r, w / 2, h / 2);
//     ctx.beginPath();
//     ctx.moveTo(x + rr, y);
//     ctx.lineTo(x + w - rr, y);
//     ctx.quadraticCurveTo(x + w, y, x + w, y + rr);
//     ctx.lineTo(x + w, y + h - rr);
//     ctx.quadraticCurveTo(x + w, y + h, x + w - rr, y + h);
//     ctx.lineTo(x + rr, y + h);
//     ctx.quadraticCurveTo(x, y + h, x, y + h - rr);
//     ctx.lineTo(x, y + rr);
//     ctx.quadraticCurveTo(x, y, x + rr, y);
//     ctx.closePath();
//   }

//   function draw() {
//     requestAnimationFrame(draw);

//     analyser.getByteFrequencyData(freqData);

//     // Recompute bands if fftSize or sampleRate changed dynamically
//     if (logBands.length !== barCount) {
//       logBands = makeLogBands(barCount, freqData.length, audioCtx.sampleRate);
//     }

//     // Per-band level and global loudness
//     const bandLevels = new Array(barCount).fill(0);
//     let globalLevelAccum = 0;

//     for (let i = 0; i < barCount; i++) {
//       const [b0, b1] = logBands[i];
//       let sum = 0;
//       const len = Math.max(1, b1 - b0 + 1);
//       for (let b = b0; b <= b1; b++) {
//         sum += magToLevel(freqData[b]);
//       }
//       const avg = sum / len;                   // 0..1
//       const gated = Math.max(0, avg - noiseFloor) / (1 - noiseFloor);
//       bandLevels[i] = gated;
//       globalLevelAccum += gated;
//     }

//     // Overall loudness 0..1
//     const globalLevel = globalLevelAccum / barCount;

//     // Number of visibly "active" bars grows with loudness
//     const minActive = Math.ceil(barCount * 0.35); // ensure many bars move at normal speech
//     const maxActive = barCount;
//     const activeBars = Math.min(
//       maxActive,
//       Math.max(minActive, Math.round(minActive + globalLevel * (maxActive - minActive)))
//     );

//     // Render parameters
//     const maxHeight = cssH * 0.8;
//     const minHeight = 3;

//     // Slight emphasis on active bars so they clearly move; others still move but less
//     const boost = 0.85 + 0.15 * globalLevel;

//     for (let i = 0; i < barCount; i++) {
//       const isActive = i < activeBars; // left-to-right activation; change to center-out if desired
//       const level = isActive ? Math.min(1, bandLevels[i] * boost) : bandLevels[i] * 0.6;
//       const target = Math.max(minHeight, level * maxHeight);
//       barHeights[i] = smoothing * barHeights[i] + (1 - smoothing) * target;
//     }

//     // Clear and paint background
//     ctx.clearRect(0, 0, cssW, cssH);
//     if (bgColor !== 'transparent') {
//       ctx.fillStyle = bgColor;
//       ctx.fillRect(0, 0, cssW, cssH);
//     }

//     // Draw bars centered vertically like your reference
//     const totalWidth = barCount * barWidth + (barCount - 1) * barGap;
//     const startX = Math.round((cssW - totalWidth) / 2);
//     const baseline = Math.round(cssH / 2);
//     ctx.fillStyle = barColor;

//     for (let i = 0; i < barCount; i++) {
//       const h = Math.max(2, barHeights[i]);
//       const x = startX + i * (barWidth + barGap);
//       const y = baseline - h / 2;
//       roundRect(x, y, barWidth, h, cornerRadius);
//       ctx.fill();
//     }
//   }

//   draw();

//   // Return a small controller API
//   return {
//     stop() {
//       stream.getTracks().forEach(t => t.stop());
//       audioCtx.close();
//     },
//     setColor(c) { ctx.fillStyle = (barColor = c); },
//     // Recreate with new bar count for simplicity
//     setBars(n) {
//       // Not hot-swapping here; restart recommended:
//       console.warn('Call startEqualizer with a new barCount to apply changes.');
//     }
//   };
// }


// function stopEqualizer({ canvasId = 'eq', clear = true } = {}) {
//   const canvas = document.getElementById(canvasId);
//   if (!canvas || !canvas.__eq) return;

//   const { stream, audioCtx, rafId, ctx, cssW, cssH } = canvas.__eq;

//   // 1) Stop animation
//   if (rafId) cancelAnimationFrame(rafId);

//   // 2) Stop mic tracks
//   try {
//     if (stream) {
//       stream.getTracks().forEach(t => t.stop());
//     }
//   } catch {}

//   // 3) Close audio context
//   try {
//     if (audioCtx && audioCtx.state !== 'closed') {
//       audioCtx.close();
//     }
//   } catch {}

//   // 4) Optionally clear canvas
//   if (clear && ctx) {
//     ctx.clearRect(0, 0, cssW, cssH);
//   }

//   // 5) Remove reference
//   delete canvas.__eq;
// }
async function startEqualizer({
  canvasId = 'eq',
  barCount = 12,
  barWidth = 10,
  barGap = 8,
  barColor = '#1da1f2',
  bgColor = 'transparent',
  cornerRadius = 4,
  smoothing = 0.7,
  fftSize = 2048
} = {}) {
  const canvas = document.getElementById(canvasId);
  if (!canvas) throw new Error('Canvas not found');
  canvas.style.display = '';
  const ctx = canvas.getContext('2d');

  // Handle HiDPI scaling
  const cssW = canvas.clientWidth || canvas.width;
  const cssH = canvas.clientHeight || canvas.height;
  const dpr = Math.max(1, window.devicePixelRatio || 1);
  canvas.width = Math.round(cssW * dpr);
  canvas.height = Math.round(cssH * dpr);
  ctx.scale(dpr, dpr);

  // Prepare audio input
  const stream = await navigator.mediaDevices.getUserMedia({ audio: true, video: false });
  const audioCtx = new (window.AudioContext || window.webkitAudioContext)();
  const source = audioCtx.createMediaStreamSource(stream);
  const analyser = audioCtx.createAnalyser();
  analyser.fftSize = fftSize;
  analyser.smoothingTimeConstant = 0.6;
  source.connect(analyser);

  const freqData = new Uint8Array(analyser.frequencyBinCount);
  const barHeights = new Array(barCount).fill(0);

  function makeLogBands(nBars, nBins, sampleRate = 44100) {
    const fNyquist = sampleRate / 2;
    const hzPerBin = fNyquist / nBins;
    const fMin = 80;
    const fMax = 8000;
    const bands = [];
    for (let i = 0; i < nBars; i++) {
      const t0 = i / nBars;
      const t1 = (i + 1) / nBars;
      const f0 = fMin * Math.pow(fMax / fMin, t0);
      const f1 = fMin * Math.pow(fMax / fMin, t1);
      const b0 = Math.max(0, Math.floor(f0 / hzPerBin));
      const b1 = Math.min(nBins - 1, Math.ceil(f1 / hzPerBin));
      bands.push([b0, b1]);
    }
    return bands;
  }

  function magToLevel(v, knee = 0.2) {
    const x = v / 255;
    const y = Math.log10(1 + 9 * x);
    return (y < knee) ? (y / knee) * 0.25 : 0.25 + (y - knee) * (0.75 / (1 - knee));
  }

  const noiseFloor = 0.02;
  let logBands = makeLogBands(barCount, freqData.length, audioCtx.sampleRate);

  function roundRect(x, y, w, h, r) {
    const rr = Math.min(r, w / 2, h / 2);
    ctx.beginPath();
    ctx.moveTo(x + rr, y);
    ctx.lineTo(x + w - rr, y);
    ctx.quadraticCurveTo(x + w, y, x + w, y + rr);
    ctx.lineTo(x + w, y + h - rr);
    ctx.quadraticCurveTo(x + w, y + h, x + w - rr, y + h);
    ctx.lineTo(x + rr, y + h);
    ctx.quadraticCurveTo(x, y + h, x, y + h - rr);
    ctx.lineTo(x, y + rr);
    ctx.quadraticCurveTo(x, y, x + rr, y);
    ctx.closePath();
  }

  let rafId = 0;

  function draw() {
    rafId = requestAnimationFrame(draw);

    analyser.getByteFrequencyData(freqData);

    if (logBands.length !== barCount) {
      logBands = makeLogBands(barCount, freqData.length, audioCtx.sampleRate);
    }

    const bandLevels = new Array(barCount).fill(0);
    let globalLevelAccum = 0;

    for (let i = 0; i < barCount; i++) {
      const [b0, b1] = logBands[i];
      let sum = 0;
      const len = Math.max(1, b1 - b0 + 1);
      for (let b = b0; b <= b1; b++) sum += magToLevel(freqData[b]);
      const avg = sum / len;
      const gated = Math.max(0, avg - noiseFloor) / (1 - noiseFloor);
      bandLevels[i] = gated;
      globalLevelAccum += gated;
    }

    const globalLevel = globalLevelAccum / barCount;

    const minActive = Math.ceil(barCount * 0.35);
    const maxActive = barCount;
    const activeBars = Math.min(
      maxActive,
      Math.max(minActive, Math.round(minActive + globalLevel * (maxActive - minActive)))
    );

    const maxHeight = cssH * 0.8;
    const minHeight = 3;
    const boost = 0.85 + 0.15 * globalLevel;

    for (let i = 0; i < barCount; i++) {
      const isActive = i < activeBars;
      const level = isActive ? Math.min(1, bandLevels[i] * boost) : bandLevels[i] * 0.6;
      const target = Math.max(minHeight, level * maxHeight);
      barHeights[i] = smoothing * barHeights[i] + (1 - smoothing) * target;
    }

    ctx.clearRect(0, 0, cssW, cssH);
    if (bgColor !== 'transparent') {
      ctx.fillStyle = bgColor;
      ctx.fillRect(0, 0, cssW, cssH);
    }

    const totalWidth = barCount * barWidth + (barCount - 1) * barGap;
    const startX = Math.round((cssW - totalWidth) / 2);
    const baseline = Math.round(cssH / 2);
    ctx.fillStyle = barColor;

    for (let i = 0; i < barCount; i++) {
      const h = Math.max(2, barHeights[i]);
      const x = startX + i * (barWidth + barGap);
      const y = baseline - h / 2;
      roundRect(x, y, barWidth, h, cornerRadius);
      ctx.fill();
    }
  }

  // Save handles so stopEqualizer can access them
  canvas.__eq = { stream, audioCtx, analyser, rafId: 0, ctx, cssW, cssH };

  // Start animation and keep rafId updated in the saved object
  function startLoop() {
    function loop() {
      canvas.__eq.rafId = requestAnimationFrame(loop);
      draw(); // draw will not call RAF now; loop holds the RAF
    }
    loop();
  }

  // Use a single RAF chain (loop) to have a single id to cancel
  // Adjust draw() to not call requestAnimationFrame itself, or wrap as above.
  // For minimal changes, we already set loop() to own the RAF.
  startLoop();

  return {
    stop() { stopEqualizer({ canvasId }); },
    setColor(c) { ctx.fillStyle = (barColor = c); },
    setBars(n) { console.warn('Restart startEqualizer with a new barCount to apply changes.'); }
  };
}

function stopEqualizer({ canvasId = 'eq', clear = true, hide = true } = {}) {
  const canvas = document.getElementById(canvasId);
  if (!canvas || !canvas.__eq) {
    if (hide && canvas) canvas.style.display = 'none';
    return;   
  }

  const { stream, audioCtx, rafId, ctx, cssW, cssH } = canvas.__eq;

  if (rafId) cancelAnimationFrame(rafId);

  try { if (stream) stream.getTracks().forEach(t => t.stop()); } catch {}

  try { if (audioCtx && audioCtx.state !== 'closed') audioCtx.close(); } catch {}

  if (clear && ctx) ctx.clearRect(0, 0, cssW, cssH);

  delete canvas.__eq;
}


document.getElementById('backBtn').onclick = function () {
    window.history.back();
};

</script>

